/******************************************************************************\
**
**     COMPANY:	LDW Software
**
**      MODULE:	Villager
**
** DESCRIPTION:	Represents an individual villager
**
**      AUTHOR:	Adrian Francis
**
\******************************************************************************/

#ifndef VILLAGER_H
#define VILLAGER_H

/******************************************************************************\
** LIBRARY INCLUDES
\******************************************************************************/
#include <string.h>
#include "../Game/Isola.h"
#include "ldwTypes.h"
#include "../Sound/Sound.h"
#include "AnimControl.h"
#include "Behavior.h"
#include "LikeList.h"
#include "DislikeList.h"
#include "VillagerSkills.h"
#include "VillagerBio.h"
#include "VillagerState.h"
#include "VillagerAI.h"
#include "VillagerAttachmentInterface.h"
#include "../Utils/SceneManager.h"
#include "../World/WorldPath.h"

class theGameState;
class ldwGameWindow;
class theStringManager;
class ldwLog;
class thePrizeMsgAssembler;

/******************************************************************************\
** CONSTANT DEFINITIONS
\******************************************************************************/
const int Villager_c_iWidth = 40;
const int Villager_c_iHeight = 65;
const int Villager_c_iMaxBehaviourLabelLength = 40;
const int Villager_c_iMaxAttachments = 4;
const int Villager_c_iChanceOfRandomGenes = 5;
const int Villager_c_iGravity = 3;

/******************************************************************************\
** CLASS DEFINITIONS
\******************************************************************************/
class CVillager : public CVillagerAI, public CVillagerBio, public CVillagerState, public CVillagerSkills, public CAnimControl
{
	friend class CVillagerAI;
	friend class CVillagerPlans;

public:

	//class structures
	struct SSaveState
	{
		bool				m_bExists;
		CVillagerBio		m_Bio;
		CVillagerState		m_State;
		CVillagerSkills		m_Skills;
		CLikeList			m_Likes;
		CDislikeList		m_Dislikes;
	};

	struct SAttachment
	{
		IVillagerAttachment *	m_pAttachment;
		int						m_iRef;
		ldwPoint				m_sOffset;
		unsigned int			m_uiTimeOut;
		unsigned int			m_uiAttachmentTime;
	};

private:

	//class data
	int						m_iRef;

	int						m_iXPos;
	int						m_iYPos;
	int						m_iXVel;
	int						m_iYVel;
	int						m_iXMom;
	int						m_iYMom;
	int						m_iFrameIndex;
	EAnimFrame				m_eAnimFrame;
	int						m_iAnimDirection; //1 or -1 depending if bending down or up, for example
	int						m_iDisplacementFrame;
	int						m_iDisplacementDirection;
	EScenePriority			m_eScenePriority;

	bool					m_bExists;
	bool					m_bHasFocus;
	bool					m_bIsInHand;
	bool					m_bIsBusy;

	EDirection				m_eDirection;
	EHeadDirection			m_eHeadDirection;
	EBodyPosition			m_eBodyPosition;
	ECarrying				m_eCarrying;
	
	EBehavior				m_eBehavior;
	char					m_sBehaviorLabel[Villager_c_iMaxBehaviourLabelLength];

	SAttachment				m_sAttachment[Villager_c_iMaxAttachments];
	int						m_iNumAttachments;

	CLikeList				m_Likes;
	CDislikeList			m_Dislikes;

	CWorldPath				mNavPath;	// not persisted in gamestate

	theGameState *			gs;
	ldwGameWindow *			gameWindow;
	theStringManager *		stringMgr;
	ldwLog *				log;
	thePrizeMsgAssembler *	PrzMsg;

	static EDirection		m_eDetailHeadDirection;

public:

	//class methods
	//setup
	public:		CVillager(void);
	public:		void Reset(void);
	public:		void Init(int lName, ESkill whichSkill, int howMuchSkill, EGender whatGender, int whatAge, const char * fatherName, const char * motherName, int fatherBodyType, int fatherHeadType, int motherBodyType, int motherHeadType, int forceX, int forceY, bool oldClothes);
	public:		void InitTwin(CVillager &Twin);
	//public:		void InitTotem(CVillager &Villager);
	public:		const bool SaveState(SSaveState &sSaveState);
	public:		const bool LoadState(SSaveState &sSaveState);

	// process
	public:		void DrawDetailVillager();
	public:		void DrawDetailVillagerParents();
	public:		void DrawEventVillager(int x, int y,EBodyPosition bodyPosition = eBodyPosition_Upright);

	private:	void SelectDirectionBasedOnVelocity();

	public:		void Update();

	public:		void NewBehavior(const EBehavior eBehavior, const SBehaviorData &sData = SBehaviorData());
	public:		void StartNewBehavior() { CVillagerAI::StartNewBehavior(*this); }
	public:		void ForgetPlans() { CVillagerAI::ForgetPlans(*this); }
	public:		void NextPlan() { CVillagerAI::NextPlan(*this); }

	protected:	void StartBending();
	protected:	void StartCheering();

	protected:	void StartTwirlingCW();
	protected:	void StartTwirlingCCW();
	protected:	void StartDiscoing();
	protected:	void StartDoingTheMonkey();
	protected:	void StartReveling();
	protected:	void StartLyingDown();
	protected:	void StartScrubbing();
	protected:	void StartExorcisting();
	protected:	void StartJoyTwirlingCW();
	protected:	void StartClimbing(const SClimbData sData);
	protected:	void StartDancing();
	protected:	void StartDiving(SDiveData &sData);
	protected:	void StartEmbrace();
	protected:	void StartGoing(SGoingData &sData);
	protected:	void StartHiding();
	protected:	void StartHealing();
	protected:	void StartJumping(SJumpData &sData);
	protected:	void StartNothing();
	protected:	void StartSwimming(const SSwimData sData);
	protected:	void StartWaiting(const EBodyPosition eBodyPosition, const EDirection eDirection, const EHeadDirection eHeadDirection);
	protected:	void StartWandering(SWanderingData &sData);
	protected:	void StartWorking();
	protected:	void StartDoingPushups();

	protected:	void ProcessClimbing(const SClimbData sData);
	protected:	void ProcessDancing();
	protected:	void ProcessDiving(SDiveData &sData);
	protected:	void ProcessEmbrace();
	protected:  void ProcessFollowPath(SGoingData &sPlan);
	protected:	void ProcessGoing(SGoingData &sData);
	protected:	void ProcessJumping(SJumpData sData);
	protected:	void ProcessSwimming();
	protected:	void ProcessWaiting();
	protected:	void ProcessWandering(SWanderingData &sData);

	private:	void AddFootprints();
	private:	void AddDust();

	public:		void ConceiveChildWith(CVillager *pPartner, bool gameInitPreg = false);
	public:		void Impregnate(int lastName, ESkill fatherSkillType, int fatherSkillValue, const char * fatherName, int fatherHeadType, int fatherBodyType, bool gameInitPreg = false);

	//public:		void SetVillagerPosition(ldwPoint where);
	public:		void DropVillager();
	public:		void NormVel(int speed);
	public:		void Stop();

	public:		void SetVillagerInHand();

	public:		const int ChanceOfDisease();

	public:		void AddLike(ELike eLike) { m_Likes.Add(eLike); m_Dislikes.Remove(eLike); }
	public:		void RemoveLike(ELike eLike) { m_Likes.Remove(eLike); }
	public:		void AddDislike(ELike eDislike) { m_Dislikes.Add(eDislike); m_Likes.Remove(eDislike); }
	public:		void RemoveDislike(ELike eDislike) { m_Dislikes.Remove(eDislike); }

	public:		const bool Attach(IVillagerAttachment &Attachment, const int iRef, int iTime = 0) { return Attach(Attachment, iRef, ldwPoint(0, 0), iTime); }
	public:		const bool Attach(IVillagerAttachment &Attachment, const int iRef, const ldwPoint sOffset, const int iTime = 0);
	public:		void Detach(IVillagerAttachment &Attachment, int &iRef);
	public:		void DetachAll();

	// access
	public:		void SetExists(bool bExists) { m_bExists = bExists; }
	public:		const bool Exists() const { return m_bExists != 0; }

	public:		void SetRef(int iRef) { m_iRef = iRef; }
	public:		const int Ref() const { return m_iRef; }
	public:		void SetFocus(bool bHasFocus) { m_bHasFocus = bHasFocus; }
	public:		const bool HasFocus() const { return m_bHasFocus != 0; }

	public:		void SetInHand(bool bIsInHand) { m_bIsInHand = bIsInHand; }
	public:		const bool IsInHand() const { return m_bIsInHand != 0; }

	public:		void SetBusy(bool bIsBusy) { m_bIsBusy = bIsBusy; }
	public:		const bool IsBusy() const { return m_bIsBusy != 0; }

	public:		const bool Likes(ELike eLike) { return m_Likes.Contains(eLike); }
	public:		const bool Dislikes(ELike eDislike) { return m_Dislikes.Contains(eDislike); }
	public:		const char *GetLikesAsString() { return m_Likes.GetListAsString(); }
	public:		const char *GetDislikesAsString() { return m_Dislikes.GetListAsString(); }

	public:		const EBehavior CurrentBehavior() const { return m_eBehavior; }
	public:		void SetBehaviorLabel(char *label) { strncpy(m_sBehaviorLabel, label, Villager_c_iMaxBehaviourLabelLength - 1); }
	public:		void SetBehaviorLabel(StringId eString) { strncpy(m_sBehaviorLabel, theStringManager::Get()->GetString(eString), Villager_c_iMaxBehaviourLabelLength - 1); }
	public:		const char *BehaviorLabel() const { return m_sBehaviorLabel; }

	public:		const float Scale() const;
	public:		const ldwPoint Dimensions() const;

	public:		void SetXPos(int iXPos) { m_iXPos = iXPos; }
	protected:	void AdjustXPos(int iAdj) { m_iXPos += iAdj; }
	public:		const int XPos() const { return m_iXPos; }
	public:		void SetYPos(int iYPos) { m_iYPos = iYPos; }
	protected:	void AdjustYPos(int iAdj) { m_iYPos += iAdj; }
	public:		const int YPos() const { return m_iYPos; }
	public:		void SetPos(const ldwPoint sPos) { m_iXPos = sPos.x, m_iYPos = sPos.y; }
	public:		const ldwPoint Pos() const { return ldwPoint(m_iXPos, m_iYPos); }
	public:		void SetFeetPos(const ldwPoint sPos);
	public:     const ldwPoint FeetPos() const;
	protected:	void SetXVel(int iXVel) { m_iXVel = iXVel; }
	protected:	void SetYVel(int iYVel) { m_iYVel = iYVel; }
	public:		const EAnimFrame AnimFrame() const { return m_eAnimFrame; }
	public:		const int DisplacementFrame() const { return m_iDisplacementFrame; }
	public:		void SetScenePriority(const EScenePriority eScenePriority) { m_eScenePriority = eScenePriority; }
	public:		const EScenePriority ScenePriority() { return m_eScenePriority; }

	public:		const EDirection Direction() const { return m_eDirection; }
	public:		void SetHeadDirection(const EHeadDirection eDirection) { if (AnimManager.IsValidHeadDirection(eDirection, m_eDirection)) m_eHeadDirection = eDirection; }
	public:		const EHeadDirection HeadDirection() const { return m_eHeadDirection; }
	protected:	void SetBodyPosition(const EBodyPosition eBodyPosition) { m_eBodyPosition = eBodyPosition; }
	public:		const EBodyPosition BodyPosition() const { return m_eBodyPosition; }

	protected:	void ResetToNeutralBodyPosition();

	public:		void SetCarrying(ECarrying eCarrying);
	public:		void ForceCarrying(ECarrying eCarrying);
	public:		const ECarrying Carrying() const { return m_eCarrying; }

	public:		const bool HasAttachment(IVillagerAttachment &Attachment, int iRef) const;

	public:		void SayOh();
};

#endif

/******************************************************************************\
** END OF FILE
\******************************************************************************/